// Configuração da API
const API_BASE_URL = './api';

// Elementos do DOM
const productExitForm = document.getElementById('product-exit-form');
const generateReportBtn = document.getElementById('generate-report');
const reportDateInput = document.getElementById('report-date');
const dailyReportDiv = document.getElementById('daily-report');
const exitsListDiv = document.getElementById('exits-list');
const messageModal = document.getElementById('message-modal');
const modalMessage = document.getElementById('modal-message');
const closeModal = document.querySelector('.close');

// Inicialização
document.addEventListener('DOMContentLoaded', function() {
    // Definir data atual nos campos de data
    const today = new Date().toISOString().split('T')[0];
    const now = new Date().toTimeString().split(' ')[0].substring(0, 5);
    
    document.getElementById('exit-date').value = today;
    document.getElementById('exit-time').value = now;
    document.getElementById('report-date').value = today;
    
    // Carregar dados iniciais
    loadRecentExits();
    generateDailyReport();
    
    // Event listeners
    productExitForm.addEventListener('submit', handleProductExitSubmit);
    generateReportBtn.addEventListener('click', generateDailyReport);
    closeModal.addEventListener('click', hideModal);
    
    // Fechar modal ao clicar fora dele
    window.addEventListener('click', function(event) {
        if (event.target === messageModal) {
            hideModal();
        }
    });
});

// Função para submeter nova saída de produto
async function handleProductExitSubmit(event) {
    event.preventDefault();
    
    const formData = new FormData(productExitForm);
    const data = {
        product_name: formData.get('product_name'),
        quantity: parseInt(formData.get('quantity')),
        exit_date: formData.get('exit_date') || undefined,
        exit_time: formData.get('exit_time') || undefined
    };
    
    try {
        showLoading(productExitForm);
        
        const response = await fetch(`${API_BASE_URL}/product_exits.php`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify(data)
        });
        
        const result = await response.json();
        
        if (response.ok) {
            showMessage('Saída de produto registada com sucesso!', 'success');
            productExitForm.reset();
            
            // Redefinir data e hora atuais
            const today = new Date().toISOString().split('T')[0];
            const now = new Date().toTimeString().split(' ')[0].substring(0, 5);
            document.getElementById('exit-date').value = today;
            document.getElementById('exit-time').value = now;
            
            // Recarregar dados
            loadRecentExits();
            generateDailyReport();
        } else {
            showMessage(`Erro: ${result.error}`, 'error');
        }
    } catch (error) {
        showMessage(`Erro de conexão: ${error.message}`, 'error');
    } finally {
        hideLoading(productExitForm);
    }
}

// Função para carregar saídas recentes
async function loadRecentExits() {
    try {
        showLoading(exitsListDiv);
        
        const response = await fetch(`${API_BASE_URL}/product_exits.php`);
        const exits = await response.json();
        
        if (response.ok) {
            displayExits(exits);
        } else {
            showMessage(`Erro ao carregar saídas: ${exits.error}`, 'error');
        }
    } catch (error) {
        showMessage(`Erro de conexão: ${error.message}`, 'error');
    } finally {
        hideLoading(exitsListDiv);
    }
}

// Função para exibir lista de saídas
function displayExits(exits) {
    if (exits.length === 0) {
        exitsListDiv.innerHTML = `
            <div class="empty-state">
                <h3>Nenhuma saída registada</h3>
                <p>Comece por registar a primeira saída de produto.</p>
            </div>
        `;
        return;
    }
    
    const exitsHTML = exits.map(exit => `
        <div class="exit-item">
            <div class="exit-product">${exit.product_name}</div>
            <div class="exit-quantity">${exit.quantity}</div>
            <div class="exit-date">${formatDate(exit.exit_date)}</div>
            <div class="exit-time">${exit.exit_time}</div>
        </div>
    `).join('');
    
    exitsListDiv.innerHTML = exitsHTML;
}

// Função para gerar relatório diário
async function generateDailyReport() {
    const reportDate = reportDateInput.value;
    
    try {
        showLoading(dailyReportDiv);
        
        const url = reportDate 
            ? `${API_BASE_URL}/report.php?date=${reportDate}`
            : `${API_BASE_URL}/report.php`;
            
        const response = await fetch(url);
        const report = await response.json();
        
        if (response.ok) {
            displayDailyReport(report);
        } else {
            showMessage(`Erro ao gerar relatório: ${report.error}`, 'error');
        }
    } catch (error) {
        showMessage(`Erro de conexão: ${error.message}`, 'error');
    } finally {
        hideLoading(dailyReportDiv);
    }
}

// Função para exibir relatório diário
function displayDailyReport(report) {
    if (report.total_exits === 0) {
        dailyReportDiv.innerHTML = `
            <div class="empty-state">
                <h3>Nenhuma saída registada</h3>
                <p>Não há saídas de produtos para ${formatDate(report.date)}.</p>
            </div>
        `;
        return;
    }
    
    // Resumo
    const summaryHTML = `
        <div class="report-summary">
            <div class="summary-card">
                <h3>${report.total_exits}</h3>
                <p>Total de Saídas</p>
            </div>
            <div class="summary-card">
                <h3>${report.total_quantity}</h3>
                <p>Quantidade Total</p>
            </div>
            <div class="summary-card">
                <h3>${Object.keys(report.products).length}</h3>
                <p>Produtos Diferentes</p>
            </div>
        </div>
    `;
    
    // Tabela de produtos
    const productsTableHTML = `
        <h3>Resumo por Produto</h3>
        <table class="products-table">
            <thead>
                <tr>
                    <th>Produto</th>
                    <th>Quantidade Total</th>
                </tr>
            </thead>
            <tbody>
                ${Object.entries(report.products).map(([product, quantity]) => `
                    <tr>
                        <td>${product}</td>
                        <td>${quantity}</td>
                    </tr>
                `).join('')}
            </tbody>
        </table>
    `;
    
    // Detalhes das saídas
    const exitsDetailsHTML = `
        <h3>Detalhes das Saídas</h3>
        <div class="exits-list">
            ${report.exits.map(exit => `
                <div class="exit-item">
                    <div class="exit-product">${exit.product_name}</div>
                    <div class="exit-quantity">${exit.quantity}</div>
                    <div class="exit-date">${formatDate(exit.exit_date)}</div>
                    <div class="exit-time">${exit.exit_time}</div>
                </div>
            `).join('')}
        </div>
    `;
    
    dailyReportDiv.innerHTML = `
        <h3>Relatório de ${formatDate(report.date)}</h3>
        ${summaryHTML}
        ${productsTableHTML}
        ${exitsDetailsHTML}
    `;
}

// Funções utilitárias
function formatDate(dateString) {
    const date = new Date(dateString + 'T00:00:00');
    return date.toLocaleDateString('pt-PT', {
        day: '2-digit',
        month: '2-digit',
        year: 'numeric'
    });
}

function showMessage(message, type = 'info') {
    modalMessage.textContent = message;
    messageModal.style.display = 'block';
    
    // Adicionar classe de tipo se necessário
    if (type === 'success') {
        messageModal.querySelector('.modal-content').classList.add('success');
    } else if (type === 'error') {
        messageModal.querySelector('.modal-content').classList.add('error');
    }
}

function hideModal() {
    messageModal.style.display = 'none';
    // Remover classes de tipo
    messageModal.querySelector('.modal-content').classList.remove('success', 'error');
}

function showLoading(element) {
    element.classList.add('loading');
}

function hideLoading(element) {
    element.classList.remove('loading');
}

// Função para atualizar automaticamente os dados a cada 30 segundos
setInterval(() => {
    loadRecentExits();
}, 30000);

