// Configuração da API
const API_BASE_URL = './api';

// Elementos do DOM
const productExitForm = document.getElementById('product-exit-form');
const generateReportBtn = document.getElementById('generate-report');
const reportDateInput = document.getElementById('report-date');
const dailyReportDiv = document.getElementById('daily-report');
const exitsListDiv = document.getElementById('exits-list');
const messageModal = document.getElementById('message-modal');
const modalMessage = document.getElementById('modal-message');
const closeModal = document.querySelector('.close');
const productSelect = document.getElementById('product-select'); // Novo elemento para seleção de produto
const productNameInput = document.getElementById('product-name'); // Campo de texto para nome do produto
const modeSelectProductBtn = document.getElementById('mode-select-product');
const modeManualInputBtn = document.getElementById('mode-manual-input');

// Estado global
let products = []; // Para armazenar os produtos carregados

// Inicialização
document.addEventListener('DOMContentLoaded', function() {
    // Verificar autenticação antes de carregar qualquer conteúdo
    checkAuthentication();
});

// Função para verificar se o utilizador está autenticado
async function checkAuthentication() {
    try {
        // Verificar autenticação de admin
        const adminResponse = await fetch(`${API_BASE_URL}/auth.php?action=check`, {
            credentials: 'include'
        });
        
        if (adminResponse.ok) {
            const adminResult = await adminResponse.json();
            if (adminResult.authenticated) {
                initializeApp();
                return;
            }
        }
        
        // Verificar autenticação de colaborador
        const colaboradorResponse = await fetch(`${API_BASE_URL}/auth_colaborador.php?action=check`, {
            credentials: 'include'
        });
        
        if (colaboradorResponse.ok) {
            const colaboradorResult = await colaboradorResponse.json();
            if (colaboradorResult.authenticated) {
                initializeApp();
                return;
            }
        }
        
        // Se não está autenticado, redirecionar para login
        redirectToLogin();
        
    } catch (error) {
        console.error('Erro ao verificar autenticação:', error);
        redirectToLogin();
    }
}

// Função para redirecionar para a página de login apropriada
function redirectToLogin() {
    // Mostrar mensagem de acesso negado
    document.body.innerHTML = `
        <div style="
            display: flex;
            justify-content: center;
            align-items: center;
            min-height: 100vh;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        ">
            <div style="
                background: white;
                border-radius: 20px;
                padding: 40px;
                box-shadow: 0 20px 60px rgba(0,0,0,0.15);
                text-align: center;
                max-width: 400px;
                width: 90%;
            ">
                <div style="font-size: 4rem; margin-bottom: 20px;">🔒</div>
                <h1 style="color: #4a5568; margin-bottom: 20px;">Acesso Restrito</h1>
                <p style="color: #718096; margin-bottom: 30px;">
                    É necessário fazer login para aceder ao sistema.
                </p>
                <div style="display: flex; gap: 15px; justify-content: center; flex-wrap: wrap;">
                    <a href="login.html" style="
                        background: linear-gradient(135deg, #dc2626 0%, #991b1b 100%);
                        color: white;
                        text-decoration: none;
                        padding: 12px 24px;
                        border-radius: 8px;
                        font-weight: 600;
                        transition: transform 0.3s ease;
                        display: inline-block;
                    " onmouseover="this.style.transform='translateY(-2px)'" onmouseout="this.style.transform='translateY(0)'">
                        Login Admin
                    </a>
                    <a href="login_colaborador.html" style="
                        background: linear-gradient(135deg, #dc2626 0%, #991b1b 100%);
                        color: white;
                        text-decoration: none;
                        padding: 12px 24px;
                        border-radius: 8px;
                        font-weight: 600;
                        transition: transform 0.3s ease;
                        display: inline-block;
                    " onmouseover="this.style.transform='translateY(-2px)'" onmouseout="this.style.transform='translateY(0)'">
                        Login Colaborador
                    </a>
                </div>
            </div>
        </div>
    `;
}

// Função para inicializar a aplicação após autenticação bem-sucedida
function initializeApp() {
    // Definir data atual nos campos de data
    const today = new Date().toISOString().split('T')[0];
    const now = new Date().toTimeString().split(' ')[0].substring(0, 5);
    
    const exitDateElement = document.getElementById('exit-date');
    const exitTimeElement = document.getElementById('exit-time');
    const reportDateElement = document.getElementById('report-date');

    if (exitDateElement) exitDateElement.value = today;
    if (exitTimeElement) exitTimeElement.value = now;
    if (reportDateElement) reportDateElement.value = today;
    
    // Carregar logo do sistema
    loadSystemLogo();
    
    // Carregar dados iniciais
    loadRecentExits();
    generateDailyReport();
    loadProductsForSelection(); // Carregar produtos para o select
    
    // Event listeners
    if (productExitForm) productExitForm.addEventListener('submit', handleProductExitSubmit);
    if (generateReportBtn) generateReportBtn.addEventListener('click', generateDailyReport);
    
    // Event listener para exportar PDF
    const exportPdfBtn = document.getElementById('export-pdf');
    if (exportPdfBtn) exportPdfBtn.addEventListener('click', exportReportToPDF);
    
    if (closeModal) closeModal.addEventListener('click', hideModal);
    
    // Fechar modal ao clicar fora dele
    window.addEventListener('click', function(event) {
        if (event.target === messageModal) {
            hideModal();
        }
    });

    // Event listeners para os botões de modo de entrada
    if (modeSelectProductBtn) {
        modeSelectProductBtn.addEventListener('click', () => setInputMode('select'));
    }
    if (modeManualInputBtn) {
        modeManualInputBtn.addEventListener('click', () => setInputMode('manual'));
    }

    // Definir modo inicial
    if (productSelect && productNameInput) {
        setInputMode('select');
    }
}

// Função para definir o modo de entrada (selecionar produto ou entrada manual)
function setInputMode(mode) {
    if (!productSelect || !productNameInput) return; // Adicionado para evitar erro se elementos não existirem

    const toggleContainer = document.querySelector('.input-mode-toggle');
    
    if (mode === 'select') {
        productSelect.style.display = 'block';
        productNameInput.style.display = 'none';
        productNameInput.removeAttribute('required');
        productSelect.setAttribute('required', 'required');
        if (modeSelectProductBtn) modeSelectProductBtn.classList.add('active');
        if (modeManualInputBtn) modeManualInputBtn.classList.remove('active');
        if (toggleContainer) toggleContainer.classList.remove('manual-active');
    } else {
        productSelect.style.display = 'none';
        productNameInput.style.display = 'block';
        productNameInput.setAttribute('required', 'required');
        productSelect.removeAttribute('required');
        if (modeSelectProductBtn) modeSelectProductBtn.classList.remove('active');
        if (modeManualInputBtn) modeManualInputBtn.classList.add('active');
        if (toggleContainer) toggleContainer.classList.add('manual-active');
    }
}

// Função para carregar produtos para o select
async function loadProductsForSelection() {
    if (!productSelect) return; // Adicionado para evitar erro se o elemento não existir

    try {
        const response = await fetch(`${API_BASE_URL}/products.php`, {
            credentials: 'include'
        });
        
        if (!response.ok) {
            throw new Error('Erro ao carregar produtos para seleção');
        }
        
        products = await response.json();
        
        productSelect.innerHTML = '<option value="">Carregando produtos...</option>';
        if (products.length > 0) {
            productSelect.innerHTML = '<option value="">Selecione um produto...</option>' +
                products.map(product => `<option value="${product.name}">${product.name} (Stock: ${product.stock})</option>`).join('');
        } else {
            productSelect.innerHTML = '<option value="">Nenhum produto disponível</option>';
        }
        
    } catch (error) {
        console.error('Erro ao carregar produtos para seleção:', error);
        showMessage('Erro ao carregar produtos para seleção: ' + error.message, 'error');
        productSelect.innerHTML = '<option value="">Erro ao carregar produtos</option>';
    }
}

// Função para submeter nova saída de produto
async function handleProductExitSubmit(event) {
    event.preventDefault();
    
    const formData = new FormData(productExitForm);
    let productName;
    if (productSelect && productSelect.style.display !== 'none') {
        productName = productSelect.value;
    } else if (productNameInput) {
        productName = productNameInput.value;
    } else {
        showMessage('Erro: Nenhum campo de produto encontrado.', 'error');
        return;
    }

    const data = {
        product_name: productName,
        quantity: parseInt(formData.get('quantity')),
        exit_date: formData.get('exit_date') || undefined,
        exit_time: formData.get('exit_time') || undefined
    };
    
    try {
        showLoading(productExitForm);
        
        const response = await fetch(`${API_BASE_URL}/product_exits.php`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify(data)
        });
        
        const result = await response.json();
        
        if (response.ok) {
            showMessage('Saída de produto registada com sucesso!', 'success');
            productExitForm.reset();
            
            // Redefinir data e hora atuais
            const today = new Date().toISOString().split('T')[0];
            const now = new Date().toTimeString().split(' ')[0].substring(0, 5);
            const exitDateElement = document.getElementById('exit-date');
            const exitTimeElement = document.getElementById('exit-time');
            if (exitDateElement) exitDateElement.value = today;
            if (exitTimeElement) exitTimeElement.value = now;
            
            // Recarregar dados
            loadRecentExits();
            generateDailyReport();
            loadProductsForSelection(); // Atualizar produtos após saída
        } else {
            showMessage(`Erro: ${result.error}`, 'error');
        }
    } catch (error) {
        showMessage(`Erro de conexão: ${error.message}`, 'error');
    } finally {
        hideLoading(productExitForm);
    }
}

// Função para carregar saídas recentes
async function loadRecentExits() {
    if (!exitsListDiv) return; // Adicionado para evitar erro se o elemento não existir
    try {
        showLoading(exitsListDiv);
        
        const response = await fetch(`${API_BASE_URL}/product_exits.php`);
        const exits = await response.json();
        
        if (response.ok) {
            displayExits(exits);
        } else {
            showMessage(`Erro ao carregar saídas: ${exits.error}`, 'error');
        }
    } catch (error) {
        showMessage(`Erro de conexão: ${error.message}`, 'error');
    } finally {
        hideLoading(exitsListDiv);
    }
}

// Função para exibir lista de saídas
function displayExits(exits) {
    if (!exitsListDiv) return; // Adicionado para evitar erro se o elemento não existir
    if (exits.length === 0) {
        exitsListDiv.innerHTML = `
            <div class="empty-state">
                <h3>Nenhuma saída registada</h3>
                <p>Comece por registar a primeira saída de produto.</p>
            </div>
        `;
        return;
    }
    
    const exitsHTML = exits.map(exit => `
        <div class="exit-item">
            <div class="exit-product">${exit.product_name}</div>
            <div class="exit-quantity">${exit.quantity}</div>
            <div class="exit-date">${formatDate(exit.exit_date)}</div>
            <div class="exit-time">${exit.exit_time}</div>
        </div>
    `).join('');
    
    exitsListDiv.innerHTML = exitsHTML;
}

// Função para gerar relatório diário
async function generateDailyReport() {
    if (!reportDateInput || !dailyReportDiv) return; // Adicionado para evitar erro se os elementos não existirem
    const reportDate = reportDateInput.value;
    
    try {
        showLoading(dailyReportDiv);
        
        const url = reportDate 
            ? `${API_BASE_URL}/report.php?date=${reportDate}`
            : `${API_BASE_URL}/report.php`;
            
        const response = await fetch(url);
        const report = await response.json();
        
        if (response.ok) {
            displayDailyReport(report);
        } else {
            showMessage(`Erro ao gerar relatório: ${report.error}`, 'error');
        }
    } catch (error) {
        showMessage(`Erro de conexão: ${error.message}`, 'error');
    } finally {
        hideLoading(dailyReportDiv);
    }
}

// Função para exibir relatório diário
function displayDailyReport(report) {
    if (!dailyReportDiv) return; // Adicionado para evitar erro se o elemento não existir
    if (report.total_exits === 0) {
        dailyReportDiv.innerHTML = `
            <div class="empty-state">
                <h3>Nenhuma saída registada</h3>
                <p>Não há saídas de produtos para ${formatDate(report.date)}.</p>
            </div>
        `;
        return;
    }
    
    // Resumo
    const summaryHTML = `
        <div class="report-summary">
            <div class="summary-card">
                <h3>${report.total_exits}</h3>
                <p>Total de Saídas</p>
            </div>
            <div class="summary-card">
                <h3>${report.total_quantity}</h3>
                <p>Quantidade Total</p>
            </div>
            <div class="summary-card">
                <h3>${Object.keys(report.products).length}</h3>
                <p>Produtos Diferentes</p>
            </div>
        </div>
    `;
    
    // Tabela de produtos
    const productsTableHTML = `
        <h3>Resumo por Produto</h3>
        <table class="products-table">
            <thead>
                <tr>
                    <th>Produto</th>
                    <th>Quantidade Total</th>
                </tr>
            </thead>
            <tbody>
                ${Object.entries(report.products).map(([product, quantity]) => `
                    <tr>
                        <td>${product}</td>
                        <td>${quantity}</td>
                    </tr>
                `).join('')}
            </tbody>
        </table>
    `;
    
    // Detalhes das saídas
    const exitsDetailsHTML = `
        <h3>Detalhes das Saídas</h3>
        <div class="exits-list">
            ${report.exits.map(exit => `
                <div class="exit-item">
                    <div class="exit-product">${exit.product_name}</div>
                    <div class="exit-quantity">${exit.quantity}</div>
                    <div class="exit-date">${formatDate(exit.exit_date)}</div>
                    <div class="exit-time">${exit.exit_time}</div>
                </div>
            `).join('')}
        </div>
    `;
    
    dailyReportDiv.innerHTML = `
        <h3>Relatório de ${formatDate(report.date)}</h3>
        ${summaryHTML}
        ${productsTableHTML}
        ${exitsDetailsHTML}
    `;
}

// Funções utilitárias
function formatDate(dateString) {
    const date = new Date(dateString + 'T00:00:00');
    return date.toLocaleDateString('pt-PT', {
        day: '2-digit',
        month: '2-digit',
        year: 'numeric'
    });
}

function showMessage(message, type = 'info') {
    if (!messageModal || !modalMessage) return; // Adicionado para evitar erro se os elementos não existirem
    modalMessage.textContent = message;
    messageModal.style.display = 'block';
    
    // Adicionar classe de tipo se necessário
    if (type === 'success') {
        messageModal.querySelector('.modal-content').classList.add('success');
    } else if (type === 'error') {
        modalMessage.classList.add('error'); // Adiciona a classe 'error' diretamente ao modalMessage
    } else {
        modalMessage.classList.remove('success', 'error'); // Remove as classes se for 'info' ou outro
    }
}

function hideModal() {
    if (!messageModal || !modalMessage) return; // Adicionado para evitar erro se os elementos não existirem
    messageModal.style.display = 'none';
    // Remover classes de tipo
    modalMessage.classList.remove('success', 'error'); // Remove as classes do modalMessage
}

function showLoading(element) {
    if (element) element.classList.add('loading');
}

function hideLoading(element) {
    if (element) element.classList.remove('loading');
}

// Função para atualizar automaticamente os dados a cada 30 segundos
setInterval(() => {
    loadRecentExits();
}, 30000);





// Função para carregar logo do sistema
async function loadSystemLogo() {
    try {
        const response = await fetch(`${API_BASE_URL}/logo.php`, {
            credentials: 'include'
        });
        
        const result = await response.json();
        const logoContainer = document.getElementById('logo-container');
        
        if (result.success && result.exists && result.logo_url) {
            logoContainer.innerHTML = `
                <img src="${result.logo_url}?t=${Date.now()}" alt="Logo do Sistema" style="max-height: 80px; max-width: 250px; margin-bottom: 10px;">
            `;
        } else {
            // Se não há logo, não mostrar nada (container fica vazio)
            logoContainer.innerHTML = '';
        }
    } catch (error) {
        console.error('Erro ao carregar logo do sistema:', error);
        // Em caso de erro, não mostrar nada
        const logoContainer = document.getElementById('logo-container');
        if (logoContainer) {
            logoContainer.innerHTML = '';
        }
    }
}



// ========== FUNÇÃO DE EXPORTAÇÃO PDF ==========

// Função para exportar relatório em PDF
async function exportReportToPDF() {
    const reportDate = document.getElementById('report-date').value;
    
    if (!reportDate) {
        showMessage('Por favor, selecione uma data para o relatório.', 'error');
        return;
    }
    
    try {
        // Mostrar indicador de carregamento
        const exportBtn = document.getElementById('export-pdf');
        const originalText = exportBtn.textContent;
        exportBtn.textContent = '⏳ Gerando PDF...';
        exportBtn.disabled = true;
        
        // Fazer requisição para gerar PDF
        const response = await fetch(`${API_BASE_URL}/pdf_report.php?date=${reportDate}`, {
            credentials: 'include'
        });
        
        if (response.ok) {
            // Criar blob do PDF
            const blob = await response.blob();
            
            // Criar URL temporário para download
            const url = window.URL.createObjectURL(blob);
            
            // Criar link temporário para download
            const a = document.createElement('a');
            a.style.display = 'none';
            a.href = url;
            a.download = `relatorio_${reportDate}.pdf`;
            
            // Adicionar ao DOM, clicar e remover
            document.body.appendChild(a);
            a.click();
            window.URL.revokeObjectURL(url);
            document.body.removeChild(a);
            
            showMessage('Relatório PDF gerado com sucesso!', 'success');
        } else {
            const errorData = await response.json();
            showMessage(errorData.error || 'Erro ao gerar relatório PDF.', 'error');
        }
    } catch (error) {
        console.error('Erro ao exportar PDF:', error);
        showMessage('Erro de conexão ao gerar PDF.', 'error');
    } finally {
        // Restaurar botão
        const exportBtn = document.getElementById('export-pdf');
        exportBtn.textContent = originalText;
        exportBtn.disabled = false;
    }
}

