<?php
/**
 * Endpoint para gestão de utilizadores
 * Apenas administradores podem aceder a este endpoint
 * Requer autenticação de administrador
 */

// Iniciar sessão
session_start();

// Configurar cabeçalhos CORS
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
header('Access-Control-Allow-Credentials: true');

// Responder a requisições OPTIONS (preflight)
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Verificar se é administrador
function isAdmin() {
    return isset($_SESSION['user_id']) && isset($_SESSION['role']) && $_SESSION['role'] === 'admin';
}

if (!isAdmin()) {
    http_response_code(403);
    echo json_encode(['error' => 'Acesso negado. Apenas administradores podem aceder a esta funcionalidade.']);
    exit();
}

require_once 'db.php';

// Verificar se o utilizador está autenticado e é admin
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    http_response_code(403);
    echo json_encode(['error' => 'Acesso negado. Apenas administradores podem gerir utilizadores.']);
    exit();
}

try {
    $db = new Database();
    $method = $_SERVER['REQUEST_METHOD'];
    
    switch ($method) {
        case 'GET':
            // Listar todos os utilizadores
            $users = $db->getAllUsers();
            echo json_encode($users);
            break;
            
        case 'POST':
            // Adicionar novo utilizador
            $input = json_decode(file_get_contents('php://input'), true);
            
            // Validar dados de entrada
            if (!isset($input['username']) || empty(trim($input['username']))) {
                http_response_code(400);
                echo json_encode(['error' => 'Nome de utilizador é obrigatório']);
                exit();
            }
            
            if (!isset($input['password']) || empty(trim($input['password']))) {
                http_response_code(400);
                echo json_encode(['error' => 'Palavra-passe é obrigatória']);
                exit();
            }
            
            if (!isset($input['role']) || !in_array($input['role'], ['admin', 'cozinha', 'lanchonete'])) {
                http_response_code(400);
                echo json_encode(['error' => 'Papel inválido. Use: admin, cozinha ou lanchonete']);
                exit();
            }
            
            // Sanitizar dados
            $username = trim($input['username']);
            $password = trim($input['password']);
            $role = $input['role'];
            
            // Validar comprimento da palavra-passe
            if (strlen($password) < 6) {
                http_response_code(400);
                echo json_encode(['error' => 'A palavra-passe deve ter pelo menos 6 caracteres']);
                exit();
            }
            
            // Adicionar utilizador
            $result = $db->addUser($username, $password, $role);
            
            http_response_code(201);
            echo json_encode($result);
            break;
            
        case 'DELETE':
            // Remover utilizador
            $user_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
            
            if ($user_id <= 0) {
                http_response_code(400);
                echo json_encode(['error' => 'ID de utilizador inválido']);
                exit();
            }
            
            // Não permitir que o admin se remova a si próprio
            if ($user_id == $_SESSION['user_id']) {
                http_response_code(400);
                echo json_encode(['error' => 'Não pode remover a sua própria conta']);
                exit();
            }
            
            $success = $db->deleteUser($user_id);
            
            if ($success) {
                echo json_encode(['success' => true, 'message' => 'Utilizador removido com sucesso']);
            } else {
                http_response_code(404);
                echo json_encode(['error' => 'Utilizador não encontrado ou não pode ser removido']);
            }
            break;
            
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Método não permitido']);
            break;
    }
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
}
?>

