<?php
/**
 * Endpoint para gestão de stock
 * Permite retornar produtos ao stock
 * Apenas admin e cozinha podem retornar produtos ao stock
 * Requer autenticação para acesso
 */

// Iniciar sessão
session_start();

// Configurar cabeçalhos CORS
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
header('Access-Control-Allow-Credentials: true');

// Responder a requisições OPTIONS (preflight)
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Verificar autenticação
function isAuthenticated() {
    return isset($_SESSION['user_id']) && isset($_SESSION['role']);
}

if (!isAuthenticated()) {
    http_response_code(401);
    echo json_encode(['error' => 'Acesso negado. É necessário fazer login.']);
    exit();
}

require_once 'db.php';

// Verificar se tem permissão para retornar ao stock
function canReturnToStock() {
    return isset($_SESSION['role']) && in_array($_SESSION['role'], ['admin', 'cozinha']);
}

try {
    $db = new Database();
    $method = $_SERVER['REQUEST_METHOD'];

// Verificar se o utilizador está autenticado
if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['error' => 'Utilizador não autenticado']);
    exit();
}

// Verificar se o utilizador tem permissão (admin ou cozinha)
$userRole = $_SESSION['role'];
if (!in_array($userRole, ['admin', 'cozinha'])) {
    http_response_code(403);
    echo json_encode(['error' => 'Acesso negado. Apenas administradores e pessoal da cozinha podem retornar produtos ao stock.']);
    exit();
}

try {
    $db = new Database();
    $method = $_SERVER['REQUEST_METHOD'];
    
    switch ($method) {
        case 'POST':
            $action = isset($_GET['action']) ? $_GET['action'] : '';
            
            if ($action === 'return') {
                // Retornar produto ao stock
                $input = json_decode(file_get_contents('php://input'), true);
                
                // Validar dados de entrada
                if (!isset($input['product_id']) || !is_numeric($input['product_id'])) {
                    http_response_code(400);
                    echo json_encode(['error' => 'ID do produto é obrigatório e deve ser numérico']);
                    exit();
                }
                
                if (!isset($input['quantity']) || !is_numeric($input['quantity']) || $input['quantity'] <= 0) {
                    http_response_code(400);
                    echo json_encode(['error' => 'Quantidade deve ser um número positivo']);
                    exit();
                }
                
                $product_id = (int)$input['product_id'];
                $quantity = (int)$input['quantity'];
                $reason = isset($input['reason']) ? trim($input['reason']) : 'Retorno ao stock';
                
                // Verificar se o produto existe
                $product = $db->getProductById($product_id);
                if (!$product) {
                    http_response_code(404);
                    echo json_encode(['error' => 'Produto não encontrado']);
                    exit();
                }
                
                // Atualizar o stock do produto
                $updatedProduct = $db->updateProductStock($product_id, $quantity);
                
                // Registar a operação de retorno (opcional - para auditoria)
                $db->logStockReturn($product_id, $quantity, $_SESSION['user_id'], $reason);
                
                echo json_encode([
                    'success' => true,
                    'message' => "Retornado {$quantity} unidades de {$product['name']} ao stock",
                    'product' => $updatedProduct
                ]);
                
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'Ação não reconhecida. Use action=return']);
            }
            break;
            
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Método não permitido']);
            break;
    }
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
}
?>

