<?php
/**
 * Endpoint para geração de relatórios diários
 * Suporta método GET com parâmetro opcional 'date'
 * Requer autenticação para acesso
 */

// Iniciar sessão
session_start();

// Configurar cabeçalhos CORS
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
header('Access-Control-Allow-Credentials: true');

// Responder a requisições OPTIONS (preflight)
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Verificar autenticação
function isAuthenticated() {
    return isset($_SESSION['user_id']) && isset($_SESSION['role']);
}

if (!isAuthenticated()) {
    http_response_code(401);
    echo json_encode(['error' => 'Acesso negado. É necessário fazer login.']);
    exit();
}

require_once 'db.php';

try {
    $db = new Database();
    $method = $_SERVER['REQUEST_METHOD'];
    
    if ($method !== 'GET') {
        http_response_code(405);
        echo json_encode(['error' => 'Método não permitido']);
        exit();
    }
    
    // Obter data do parâmetro ou usar data atual
    $date = isset($_GET['date']) ? $_GET['date'] : date('Y-m-d');
    
    // Validar formato da data
    if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $date)) {
        http_response_code(400);
        echo json_encode(['error' => 'Formato de data inválido. Use YYYY-MM-DD']);
        exit();
    }
    
    // Validar se a data é válida
    $date_parts = explode('-', $date);
    if (!checkdate($date_parts[1], $date_parts[2], $date_parts[0])) {
        http_response_code(400);
        echo json_encode(['error' => 'Data inválida']);
        exit();
    }
    
    // Gerar relatório diário
    $report = $db->getDailyReport($date);
    
    echo json_encode($report);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
}
?>

