<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);
/**
 * Endpoint para gestão de produtos
 * Apenas administradores podem adicionar/editar/remover produtos
 * Outros utilizadores podem apenas visualizar
 * Requer autenticação para acesso
 */

// Iniciar sessão
session_start();

// Configurar cabeçalhos CORS
header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type");
header("Access-Control-Allow-Credentials: true");

// Responder a requisições OPTIONS (preflight)
if ($_SERVER["REQUEST_METHOD"] === "OPTIONS") {
    http_response_code(200);
    exit();
}

// Verificar autenticação
function isAuthenticated() {
    return isset($_SESSION["user_id"]) && isset($_SESSION["role"]);
}

if (!isAuthenticated()) {
    http_response_code(401);
    echo json_encode(["error" => "Acesso negado. É necessário fazer login."]);
    exit();
}

require_once 'db.php';

// Verificar se é administrador para operações de escrita
function isAdmin() {
    return isset($_SESSION["role"]) && $_SESSION["role"] === "admin";
}

try {
    $db = new Database();
    $method = $_SERVER["REQUEST_METHOD"];
    $isAdmin = $_SESSION["role"] === "admin";
    
    switch ($method) {
        case 'GET':
            // Listar todos os produtos (todos os utilizadores autenticados podem ver)
            $products = $db->getAllProducts();
            echo json_encode($products);
            break;
            
        case 'POST':
            // Adicionar novo produto (apenas admin)
            if (!$isAdmin) {
                http_response_code(403);
                echo json_encode(['error' => 'Acesso negado. Apenas administradores podem adicionar produtos.']);
                exit();
            }
            
            $input = json_decode(file_get_contents('php://input'), true);
            
            // Validar dados de entrada
            if (!isset($input['name']) || empty(trim($input['name']))) {
                http_response_code(400);
                echo json_encode(['error' => 'Nome do produto é obrigatório']);
                exit();
            }
            
            $name = trim($input['name']);
            $stock = isset($input['stock']) ? (int)$input['stock'] : 0;
            
            if ($stock < 0) {
                http_response_code(400);
                echo json_encode(['error' => 'Stock não pode ser negativo']);
                exit();
            }
            
            // Adicionar produto
            $result = $db->addProduct($name, $stock);
            
            http_response_code(201);
            echo json_encode($result);
            break;
            
        case 'PUT':
            // Atualizar produto (apenas admin)
            if (!$isAdmin) {
                http_response_code(403);
                echo json_encode(['error' => 'Acesso negado. Apenas administradores podem editar produtos.']);
                exit();
            }
            
            $product_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
            
            if ($product_id <= 0) {
                http_response_code(400);
                echo json_encode(['error' => 'ID de produto inválido']);
                exit();
            }
            
            $input = json_decode(file_get_contents('php://input'), true);
            
            // Validar dados de entrada
            if (!isset($input['name']) || empty(trim($input['name']))) {
                http_response_code(400);
                echo json_encode(['error' => 'Nome do produto é obrigatório']);
                exit();
            }
            
            $name = trim($input['name']);
            $stock = isset($input['stock']) ? (int)$input['stock'] : 0;
            
            if ($stock < 0) {
                http_response_code(400);
                echo json_encode(['error' => 'Stock não pode ser negativo']);
                exit();
            }
            
            // Atualizar produto
            $result = $db->updateProduct($product_id, $name, $stock);
            
            if ($result) {
                echo json_encode($result);
            } else {
                http_response_code(404);
                echo json_encode(['error' => 'Produto não encontrado']);
            }
            break;
            
        case 'DELETE':
            // Remover produto (apenas admin)
            if (!$isAdmin) {
                http_response_code(403);
                echo json_encode(['error' => 'Acesso negado. Apenas administradores podem remover produtos.']);
                exit();
            }
            
            $product_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
            
            if ($product_id <= 0) {
                http_response_code(400);
                echo json_encode(['error' => 'ID de produto inválido']);
                exit();
            }
            
            $success = $db->deleteProduct($product_id);
            
            if ($success) {
                echo json_encode(['success' => true, 'message' => 'Produto removido com sucesso']);
            } else {
                http_response_code(404);
                echo json_encode(['error' => 'Produto não encontrado']);
            }
            break;
            
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Método não permitido']);
            break;
    }
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
}
?>

