<?php
/**
 * Endpoint para gestão de saídas de produtos
 * Suporta métodos GET (listar) e POST (adicionar)
 * Requer autenticação para acesso
 */

// Iniciar sessão
session_start();

// Configurar cabeçalhos CORS
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
header('Access-Control-Allow-Credentials: true');

// Responder a requisições OPTIONS (preflight)
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Verificar autenticação
function isAuthenticated() {
    return isset($_SESSION['user_id']) && isset($_SESSION['role']);
}

if (!isAuthenticated()) {
    http_response_code(401);
    echo json_encode(['error' => 'Acesso negado. É necessário fazer login.']);
    exit();
}

require_once 'db.php';

try {
    $db = new Database();
    $method = $_SERVER['REQUEST_METHOD'];
    
    switch ($method) {
        case 'GET':
            // Listar todas as saídas de produtos
            $exits = $db->getAllProductExits();
            echo json_encode($exits);
            break;
            
        case 'POST':
            // Adicionar nova saída de produto
            $input = json_decode(file_get_contents('php://input'), true);
            
            // Verificar se é saída por produto_id (com controlo de stock) ou product_name (modo antigo)
            if (isset($input['product_id']) && is_numeric($input['product_id'])) {
                // Modo novo: usar produto da tabela products com controlo de stock
                $product_id = (int)$input['product_id'];
                
                if (!isset($input['quantity']) || !is_numeric($input['quantity']) || $input['quantity'] <= 0) {
                    http_response_code(400);
                    echo json_encode(['error' => 'Quantidade deve ser um número positivo']);
                    exit();
                }
                
                $quantity = (int)$input['quantity'];
                $exit_date = isset($input['exit_date']) && !empty($input['exit_date']) ? $input['exit_date'] : null;
                $exit_time = isset($input['exit_time']) && !empty($input['exit_time']) ? $input['exit_time'] : null;
                
                // Validar formato da data se fornecida
                if ($exit_date && !preg_match('/^\d{4}-\d{2}-\d{2}$/', $exit_date)) {
                    http_response_code(400);
                    echo json_encode(['error' => 'Formato de data inválido. Use YYYY-MM-DD']);
                    exit();
                }
                
                // Validar formato da hora se fornecida
                if ($exit_time && !preg_match('/^\d{2}:\d{2}(:\d{2})?$/', $exit_time)) {
                    http_response_code(400);
                    echo json_encode(['error' => 'Formato de hora inválido. Use HH:MM ou HH:MM:SS']);
                    exit();
                }
                
                // Adicionar saída de produto com controlo de stock
                $result = $db->addProductExitFromStock($product_id, $quantity, $exit_date, $exit_time);
                
            } else {
                // Modo antigo: usar product_name (sem controlo de stock)
                if (!isset($input['product_name']) || empty(trim($input['product_name']))) {
                    http_response_code(400);
                    echo json_encode(['error' => 'Nome do produto é obrigatório']);
                    exit();
                }
                
                if (!isset($input['quantity']) || !is_numeric($input['quantity']) || $input['quantity'] <= 0) {
                    http_response_code(400);
                    echo json_encode(['error' => 'Quantidade deve ser um número positivo']);
                    exit();
                }
                
                // Sanitizar dados
                $product_name = trim($input['product_name']);
                $quantity = (int)$input['quantity'];
                $exit_date = isset($input['exit_date']) && !empty($input['exit_date']) ? $input['exit_date'] : null;
                $exit_time = isset($input['exit_time']) && !empty($input['exit_time']) ? $input['exit_time'] : null;
                
                // Validar formato da data se fornecida
                if ($exit_date && !preg_match('/^\d{4}-\d{2}-\d{2}$/', $exit_date)) {
                    http_response_code(400);
                    echo json_encode(['error' => 'Formato de data inválido. Use YYYY-MM-DD']);
                    exit();
                }
                
                // Validar formato da hora se fornecida
                if ($exit_time && !preg_match('/^\d{2}:\d{2}(:\d{2})?$/', $exit_time)) {
                    http_response_code(400);
                    echo json_encode(['error' => 'Formato de hora inválido. Use HH:MM ou HH:MM:SS']);
                    exit();
                }
                
                // Adicionar saída de produto (modo antigo)
                $result = $db->addProductExit($product_name, $quantity, $exit_date, $exit_time);
            }
            
            http_response_code(201);
            echo json_encode($result);
            break;
            
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Método não permitido']);
            break;
    }
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
}
?>

