<?php
// Configurações de CORS
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Responder a requisições OPTIONS (preflight)
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Iniciar sessão
session_start();

// Incluir conexão com base de dados
require_once 'db.php';

// Incluir TCPDF
require_once '../tcpdf/tcpdf.php';

// Verificar autenticação
if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['error' => 'Não autenticado']);
    exit();
}

try {
    $db = new Database();
    $pdo = $db->getConnection();
    
    if ($_SERVER['REQUEST_METHOD'] === 'GET') {
        $date = $_GET['date'] ?? date('Y-m-d');
        
        // Buscar saídas do dia
        $stmt = $pdo->prepare("
            SELECT product_name, quantity, exit_date, exit_time 
            FROM product_exits 
            WHERE exit_date = ? 
            ORDER BY exit_time ASC
        ");
        $stmt->execute([$date]);
        $exits = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Calcular estatísticas
        $totalExits = count($exits);
        $totalQuantity = array_sum(array_column($exits, 'quantity'));
        
        // Contar produtos únicos
        $uniqueProducts = array_unique(array_column($exits, 'product_name'));
        $uniqueProductsCount = count($uniqueProducts);
        
        // Resumo por produto
        $productSummary = [];
        foreach ($exits as $exit) {
            $product = $exit['product_name'];
            if (!isset($productSummary[$product])) {
                $productSummary[$product] = 0;
            }
            $productSummary[$product] += $exit['quantity'];
        }
        
        // Gerar PDF usando TCPDF
        $pdf = generatePDF($date, $exits, $totalExits, $totalQuantity, $uniqueProductsCount, $productSummary);
        
        // Configurar cabeçalhos para download de PDF
        header('Content-Type: application/pdf');
        header('Content-Disposition: attachment; filename="relatorio_' . $date . '.pdf"');
        header('Cache-Control: private, max-age=0, must-revalidate');
        header('Pragma: public');
        
        // Enviar PDF
        echo $pdf;
        
    } else {
        http_response_code(405);
        echo json_encode(['error' => 'Método não permitido']);
    }
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Erro interno do servidor: ' . $e->getMessage()]);
}

function generatePDF($date, $exits, $totalExits, $totalQuantity, $uniqueProductsCount, $productSummary) {
    $formattedDate = date('d/m/Y', strtotime($date));
    
    // Criar novo PDF
    $pdf = new TCPDF(PDF_PAGE_ORIENTATION, PDF_UNIT, PDF_PAGE_FORMAT, true, 'UTF-8', false);
    
    // Configurar informações do documento
    $pdf->SetCreator('Sistema de Saída de Produtos');
    $pdf->SetAuthor('Lanchonete Management System');
    $pdf->SetTitle('Relatório de Saídas - ' . $formattedDate);
    $pdf->SetSubject('Relatório Diário de Saídas de Produtos');
    
    // Configurar margens
    $pdf->SetMargins(15, 20, 15);
    $pdf->SetHeaderMargin(10);
    $pdf->SetFooterMargin(10);
    
    // Configurar quebra automática de página
    $pdf->SetAutoPageBreak(TRUE, 25);
    
    // Adicionar página
    $pdf->AddPage();
    
    // Configurar fonte
    $pdf->SetFont('helvetica', '', 12);
    
    // Cabeçalho
    $pdf->SetFont('helvetica', 'B', 20);
    $pdf->SetTextColor(220, 38, 38); // Vermelho
    $pdf->Cell(0, 15, '🍔 Sistema de Saída de Produtos', 0, 1, 'C');
    
    $pdf->SetFont('helvetica', 'B', 16);
    $pdf->SetTextColor(0, 0, 0); // Preto
    $pdf->Cell(0, 10, 'Relatório Diário de Saídas', 0, 1, 'C');
    
    $pdf->SetFont('helvetica', '', 12);
    $pdf->Cell(0, 8, 'Data: ' . $formattedDate, 0, 1, 'C');
    $pdf->Cell(0, 8, 'Gerado em: ' . date('d/m/Y H:i:s'), 0, 1, 'C');
    
    $pdf->Ln(10);
    
    // Linha separadora
    $pdf->SetDrawColor(220, 38, 38);
    $pdf->Line(15, $pdf->GetY(), 195, $pdf->GetY());
    $pdf->Ln(10);
    
    // Estatísticas resumo
    $pdf->SetFont('helvetica', 'B', 14);
    $pdf->SetTextColor(220, 38, 38);
    $pdf->Cell(0, 10, 'Resumo do Dia', 0, 1, 'L');
    
    $pdf->SetFont('helvetica', '', 12);
    $pdf->SetTextColor(0, 0, 0);
    
    // Criar tabela de resumo
    $pdf->SetFillColor(248, 249, 250);
    $pdf->Cell(60, 10, 'Total de Saídas:', 1, 0, 'L', true);
    $pdf->SetFont('helvetica', 'B', 12);
    $pdf->SetTextColor(220, 38, 38);
    $pdf->Cell(30, 10, $totalExits, 1, 1, 'C');
    
    $pdf->SetFont('helvetica', '', 12);
    $pdf->SetTextColor(0, 0, 0);
    $pdf->Cell(60, 10, 'Quantidade Total:', 1, 0, 'L', true);
    $pdf->SetFont('helvetica', 'B', 12);
    $pdf->SetTextColor(220, 38, 38);
    $pdf->Cell(30, 10, $totalQuantity, 1, 1, 'C');
    
    $pdf->SetFont('helvetica', '', 12);
    $pdf->SetTextColor(0, 0, 0);
    $pdf->Cell(60, 10, 'Produtos Diferentes:', 1, 0, 'L', true);
    $pdf->SetFont('helvetica', 'B', 12);
    $pdf->SetTextColor(220, 38, 38);
    $pdf->Cell(30, 10, $uniqueProductsCount, 1, 1, 'C');
    
    $pdf->Ln(10);
    
    // Resumo por produto
    if (!empty($productSummary)) {
        $pdf->SetFont('helvetica', 'B', 14);
        $pdf->SetTextColor(220, 38, 38);
        $pdf->Cell(0, 10, 'Resumo por Produto', 0, 1, 'L');
        
        // Cabeçalho da tabela
        $pdf->SetFont('helvetica', 'B', 12);
        $pdf->SetTextColor(255, 255, 255);
        $pdf->SetFillColor(220, 38, 38);
        $pdf->Cell(120, 10, 'Produto', 1, 0, 'L', true);
        $pdf->Cell(40, 10, 'Quantidade Total', 1, 1, 'C', true);
        
        // Dados da tabela
        $pdf->SetFont('helvetica', '', 11);
        $pdf->SetTextColor(0, 0, 0);
        $fill = false;
        foreach ($productSummary as $product => $quantity) {
            $pdf->SetFillColor($fill ? 248 : 255, $fill ? 249 : 255, $fill ? 250 : 255);
            $pdf->Cell(120, 8, $product, 1, 0, 'L', true);
            $pdf->SetFont('helvetica', 'B', 11);
            $pdf->SetTextColor(220, 38, 38);
            $pdf->Cell(40, 8, $quantity, 1, 1, 'C', true);
            $pdf->SetFont('helvetica', '', 11);
            $pdf->SetTextColor(0, 0, 0);
            $fill = !$fill;
        }
        
        $pdf->Ln(10);
    }
    
    // Detalhes das saídas
    if (!empty($exits)) {
        $pdf->SetFont('helvetica', 'B', 14);
        $pdf->SetTextColor(220, 38, 38);
        $pdf->Cell(0, 10, 'Detalhes das Saídas', 0, 1, 'L');
        
        // Cabeçalho da tabela
        $pdf->SetFont('helvetica', 'B', 12);
        $pdf->SetTextColor(255, 255, 255);
        $pdf->SetFillColor(220, 38, 38);
        $pdf->Cell(100, 10, 'Produto', 1, 0, 'L', true);
        $pdf->Cell(30, 10, 'Quantidade', 1, 0, 'C', true);
        $pdf->Cell(30, 10, 'Hora', 1, 1, 'C', true);
        
        // Dados da tabela
        $pdf->SetFont('helvetica', '', 11);
        $pdf->SetTextColor(0, 0, 0);
        $fill = false;
        foreach ($exits as $exit) {
            $pdf->SetFillColor($fill ? 248 : 255, $fill ? 249 : 255, $fill ? 250 : 255);
            $pdf->Cell(100, 8, $exit['product_name'], 1, 0, 'L', true);
            $pdf->SetFont('helvetica', 'B', 11);
            $pdf->SetTextColor(220, 38, 38);
            $pdf->Cell(30, 8, $exit['quantity'], 1, 0, 'C', true);
            $pdf->SetFont('helvetica', '', 11);
            $pdf->SetTextColor(0, 0, 0);
            $pdf->Cell(30, 8, $exit['exit_time'], 1, 1, 'C', true);
            $fill = !$fill;
        }
    } else {
        $pdf->SetFont('helvetica', 'I', 12);
        $pdf->SetTextColor(100, 100, 100);
        $pdf->Cell(0, 10, 'Nenhuma saída registada para esta data.', 0, 1, 'C');
    }
    
    // Rodapé
    $pdf->Ln(15);
    $pdf->SetDrawColor(220, 38, 38);
    $pdf->Line(15, $pdf->GetY(), 195, $pdf->GetY());
    $pdf->Ln(5);
    
    $pdf->SetFont('helvetica', '', 10);
    $pdf->SetTextColor(100, 100, 100);
    $pdf->Cell(0, 5, 'Relatório gerado automaticamente pelo Sistema de Saída de Produtos', 0, 1, 'C');
    $pdf->Cell(0, 5, '© ' . date('Y') . ' - Lanchonete Management System', 0, 1, 'C');
    
    return $pdf->Output('', 'S');
}
?>

