<?php
/**
 * Endpoint para gestão de logo do sistema
 * Apenas administradores podem fazer upload/alterar logo
 * Requer autenticação para acesso
 */

// Iniciar sessão
session_start();

// Configurar cabeçalhos CORS
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type");
header("Access-Control-Allow-Credentials: true");

// Responder a requisições OPTIONS (preflight)
if ($_SERVER["REQUEST_METHOD"] === "OPTIONS") {
    http_response_code(200);
    exit();
}

// Verificar autenticação
function isAuthenticated() {
    return isset($_SESSION["user_id"]) && isset($_SESSION["role"]);
}

if (!isAuthenticated()) {
    http_response_code(401);
    echo json_encode(["error" => "Acesso negado. É necessário fazer login."]);
    exit();
}

// Verificar se é administrador
function isAdmin() {
    return isset($_SESSION["role"]) && $_SESSION["role"] === "admin";
}

$method = $_SERVER["REQUEST_METHOD"];

try {
    switch ($method) {
        case 'GET':
            // Obter logo atual
            $logoPath = '../uploads/logo_sistema.png';
            if (file_exists($logoPath)) {
                header("Content-Type: application/json");
                echo json_encode([
                    'success' => true,
                    'logo_url' => 'uploads/logo_sistema.png',
                    'exists' => true
                ]);
            } else {
                header("Content-Type: application/json");
                echo json_encode([
                    'success' => true,
                    'logo_url' => null,
                    'exists' => false
                ]);
            }
            break;
            
        case 'POST':
            // Upload de nova logo (apenas admin)
            if (!isAdmin()) {
                http_response_code(403);
                echo json_encode(['error' => 'Acesso negado. Apenas administradores podem alterar a logo.']);
                exit();
            }
            
            // Verificar se foi enviado um arquivo
            if (!isset($_FILES['logo']) || $_FILES['logo']['error'] !== UPLOAD_ERR_OK) {
                http_response_code(400);
                echo json_encode(['error' => 'Nenhum arquivo de logo foi enviado ou ocorreu um erro no upload.']);
                exit();
            }
            
            $uploadedFile = $_FILES['logo'];
            
            // Verificar tipo de arquivo
            $allowedTypes = ['image/png', 'image/jpeg', 'image/jpg', 'image/gif'];
            if (!in_array($uploadedFile['type'], $allowedTypes)) {
                http_response_code(400);
                echo json_encode(['error' => 'Tipo de arquivo não permitido. Use PNG, JPG, JPEG ou GIF.']);
                exit();
            }
            
            // Verificar tamanho do arquivo (máximo 2MB)
            if ($uploadedFile['size'] > 2 * 1024 * 1024) {
                http_response_code(400);
                echo json_encode(['error' => 'Arquivo muito grande. Tamanho máximo: 2MB.']);
                exit();
            }
            
            // Definir caminho de destino
            $uploadDir = '../uploads/';
            if (!is_dir($uploadDir)) {
                mkdir($uploadDir, 0755, true);
            }
            
            $destinationPath = $uploadDir . 'logo_sistema.png';
            
            // Fazer backup da logo anterior se existir
            if (file_exists($destinationPath)) {
                $backupPath = $uploadDir . 'logo_sistema_backup_' . date('Y-m-d_H-i-s') . '.png';
                copy($destinationPath, $backupPath);
            }
            
            // Mover arquivo para destino
            if (move_uploaded_file($uploadedFile['tmp_name'], $destinationPath)) {
                header("Content-Type: application/json");
                echo json_encode([
                    'success' => true,
                    'message' => 'Logo atualizada com sucesso!',
                    'logo_url' => 'uploads/logo_sistema.png'
                ]);
            } else {
                http_response_code(500);
                echo json_encode(['error' => 'Erro ao salvar o arquivo de logo.']);
            }
            break;
            
        case 'DELETE':
            // Remover logo (apenas admin)
            if (!isAdmin()) {
                http_response_code(403);
                echo json_encode(['error' => 'Acesso negado. Apenas administradores podem remover a logo.']);
                exit();
            }
            
            $logoPath = '../uploads/logo_sistema.png';
            if (file_exists($logoPath)) {
                // Fazer backup antes de remover
                $backupPath = '../uploads/logo_sistema_removed_' . date('Y-m-d_H-i-s') . '.png';
                copy($logoPath, $backupPath);
                
                if (unlink($logoPath)) {
                    header("Content-Type: application/json");
                    echo json_encode([
                        'success' => true,
                        'message' => 'Logo removida com sucesso!'
                    ]);
                } else {
                    http_response_code(500);
                    echo json_encode(['error' => 'Erro ao remover o arquivo de logo.']);
                }
            } else {
                http_response_code(404);
                echo json_encode(['error' => 'Logo não encontrada.']);
            }
            break;
            
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Método não permitido']);
            break;
    }
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
}
?>

