<?php
/**
 * Endpoint específico para autenticação de colaboradores (não-admin)
 * Permite apenas login de utilizadores com papel 'cozinha' ou 'lanchonete'
 */

// Iniciar sessão
session_start();

// Configurar cabeçalhos CORS
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
header('Access-Control-Allow-Credentials: true');

// Responder a requisições OPTIONS (preflight)
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once 'db.php';

try {
    $db = new Database();
    $method = $_SERVER['REQUEST_METHOD'];
    $action = isset($_GET['action']) ? $_GET['action'] : '';
    
    switch ($method) {
        case 'POST':
            if ($action === 'login') {
                // Login do colaborador
                $input = json_decode(file_get_contents('php://input'), true);
                
                if (!isset($input['username']) || !isset($input['password'])) {
                    http_response_code(400);
                    echo json_encode(['error' => 'Username e password são obrigatórios']);
                    exit();
                }
                
                $user = $db->authenticateUser($input['username'], $input['password']);
                
                if ($user) {
                    // Verificar se o utilizador é colaborador (não admin)
                    if ($user['role'] === 'admin') {
                        http_response_code(403);
                        echo json_encode(['error' => 'Acesso negado. Esta página é apenas para colaboradores.']);
                        exit();
                    }
                    
                    // Verificar se o papel é válido para colaboradores
                    if (!in_array($user['role'], ['cozinha', 'lanchonete'])) {
                        http_response_code(403);
                        echo json_encode(['error' => 'Papel de utilizador não autorizado para colaboradores.']);
                        exit();
                    }
                    
                    $_SESSION['user_id'] = $user['id'];
                    $_SESSION['username'] = $user['username'];
                    $_SESSION['role'] = $user['role'];
                    
                    echo json_encode([
                        'success' => true,
                        'user' => $user,
                        'message' => 'Login de colaborador realizado com sucesso'
                    ]);
                } else {
                    http_response_code(401);
                    echo json_encode(['error' => 'Credenciais inválidas']);
                }
                
            } elseif ($action === 'logout') {
                // Logout do colaborador
                session_destroy();
                echo json_encode(['success' => true, 'message' => 'Logout realizado com sucesso']);
                
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'Ação não reconhecida']);
            }
            break;
            
        case 'GET':
            if ($action === 'check') {
                // Verificar se o colaborador está autenticado
                if (isset($_SESSION['user_id']) && in_array($_SESSION['role'], ['cozinha', 'lanchonete'])) {
                    echo json_encode([
                        'authenticated' => true,
                        'user' => [
                            'id' => $_SESSION['user_id'],
                            'username' => $_SESSION['username'],
                            'role' => $_SESSION['role']
                        ]
                    ]);
                } else {
                    echo json_encode(['authenticated' => false]);
                }
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'Ação não reconhecida']);
            }
            break;
            
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Método não permitido']);
            break;
    }
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
}

/**
 * Função auxiliar para verificar se o colaborador está autenticado
 */
function isColaboradorAuthenticated() {
    return isset($_SESSION['user_id']) && in_array($_SESSION['role'], ['cozinha', 'lanchonete']);
}

/**
 * Função auxiliar para verificar se o colaborador tem permissão para uma ação
 */
function colaboradorHasPermission($requiredRole) {
    if (!isColaboradorAuthenticated()) {
        return false;
    }
    
    $userRole = $_SESSION['role'];
    
    // Verificar permissões específicas para colaboradores
    switch ($requiredRole) {
        case 'cozinha':
            return $userRole === 'cozinha';
        case 'lanchonete':
            return in_array($userRole, ['cozinha', 'lanchonete']);
        default:
            return false;
    }
}
?>

