<?php
/**
 * Endpoint para autenticação de utilizadores
 * Suporta login, logout e verificação de sessão
 */

// Iniciar sessão
session_start();

// Configurar cabeçalhos CORS
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
header('Access-Control-Allow-Credentials: true');

// Responder a requisições OPTIONS (preflight)
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once 'db.php';

try {
    $db = new Database();
    $method = $_SERVER['REQUEST_METHOD'];
    $action = isset($_GET['action']) ? $_GET['action'] : '';
    
    switch ($method) {
        case 'POST':
            if ($action === 'login') {
                // Login do utilizador
                $input = json_decode(file_get_contents('php://input'), true);
                
                if (!isset($input['username']) || !isset($input['password'])) {
                    http_response_code(400);
                    echo json_encode(['error' => 'Username e password são obrigatórios']);
                    exit();
                }
                
                $user = $db->authenticateUser($input['username'], $input['password']);
                
                if ($user) {
                    $_SESSION['user_id'] = $user['id'];
                    $_SESSION['username'] = $user['username'];
                    $_SESSION['role'] = $user['role'];
                    
                    echo json_encode([
                        'success' => true,
                        'user' => $user
                    ]);
                } else {
                    http_response_code(401);
                    echo json_encode(['error' => 'Credenciais inválidas']);
                }
                
            } elseif ($action === 'logout') {
                // Logout do utilizador
                session_destroy();
                echo json_encode(['success' => true, 'message' => 'Logout realizado com sucesso']);
                
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'Ação não reconhecida']);
            }
            break;
            
        case 'GET':
            if ($action === 'check') {
                // Verificar se o utilizador está autenticado
                if (isset($_SESSION['user_id'])) {
                    echo json_encode([
                        'authenticated' => true,
                        'user' => [
                            'id' => $_SESSION['user_id'],
                            'username' => $_SESSION['username'],
                            'role' => $_SESSION['role']
                        ]
                    ]);
                } else {
                    echo json_encode(['authenticated' => false]);
                }
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'Ação não reconhecida']);
            }
            break;
            
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Método não permitido']);
            break;
    }
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
}

/**
 * Função auxiliar para verificar se o utilizador está autenticado
 */
function isAuthenticated() {
    return isset($_SESSION['user_id']);
}

/**
 * Função auxiliar para verificar se o utilizador é admin
 */
function isAdmin() {
    return isset($_SESSION['role']) && $_SESSION['role'] === 'admin';
}

/**
 * Função auxiliar para verificar se o utilizador tem permissão para uma ação
 */
function hasPermission($requiredRole) {
    if (!isAuthenticated()) {
        return false;
    }
    
    $userRole = $_SESSION['role'];
    
    // Admin tem acesso a tudo
    if ($userRole === 'admin') {
        return true;
    }
    
    // Verificar permissões específicas
    switch ($requiredRole) {
        case 'admin':
            return $userRole === 'admin';
        case 'cozinha':
            return in_array($userRole, ['admin', 'cozinha']);
        case 'lanchonete':
            return in_array($userRole, ['admin', 'cozinha', 'lanchonete']);
        default:
            return false;
    }
}
?>

